from django.core.serializers import serialize
from django.views.decorators.csrf import csrf_exempt
from django.http import JsonResponse
from django.conf import settings
from rest_framework import status
from rest_framework.decorators import api_view, permission_classes
from rest_framework.permissions import IsAuthenticated, AllowAny
from rest_framework.response import Response
from rest_framework.utils.representation import serializer_repr

from api_rbs.models import InfoAcount, Orders
from api_rbs.v1.serializer import ContaSerializer, HomeRequestSerializer, HomeReturnDataSerializer, \
    OpenOrdersRequestSerializer, OpenOrdersReturnDataSerializer, HistoryOrdersRequestSerializer, \
    HistoryOrdersReturnDataSerializer, ProfileUserRequestSerializer
from site_rbs.models import FbsLicenses
import pika
import json

import re

from utils.consts.api_rbs import OrderStatus


@csrf_exempt
@api_view(["POST",])
@permission_classes([AllowAny])
def receive_order(request):

    # raw_body = request.body.rstrip(b"\x00")
    # raw_str = raw_body.decode("utf-8")
    # data = json.loads(raw_str)

    data = request.data

    serializes = ContaSerializer(data=data)
    if serializes.is_valid():
        data = serializes.validated_data
        print(data)
        # publica a mensagem na fila 'received-order' do RabbitMQ
        connection = pika.BlockingConnection(
            pika.ConnectionParameters(
                host="localhost",
                port=5672,
                credentials=pika.PlainCredentials(
                    settings.RABBITMQ_USER, settings.RABBITMQ_PASSWORD
                ),
            )
        )
        channel = connection.channel()
        channel.queue_declare(queue="received-order", durable=True)
        channel.basic_publish(
            exchange="", routing_key="received-order", body=json.dumps(data)
        )

        # Retorna uma resposta de sucesso
        return JsonResponse(
            {"message": "Pedido recebido e publicado com sucesso."}, status=200
        )
    else:
        return Response(serializes.errors, status=status.HTTP_400_BAD_REQUEST)


@csrf_exempt
@api_view(["POST",])
@permission_classes([AllowAny])
def home_info(request):
    """
    View to get home information.

    <br>
     payload example:{
	    "numero_conta": "220747665"
    }

    <br>

    response example:
    {
	"fbs_lecense": 8,
	"cross_balance": "7158.75",
	"net_balance": "6451.65",
	"open_balance": "-707.10",
	"created": "2025-06-26T21:11:21.089579Z",
	"modified": "2025-06-26T21:23:50.022179Z",
	"get_licenses": [
		{
			"nomeConta": "Robson Fernandes De Souza Scavazza",
			"numeroConta": "500"
		}
	    ]
    }
    <br>
     payload example:{
	    "numero_conta": "220747665"
    }

    """
    serializers = HomeRequestSerializer(data=request.data)
    if serializers.is_valid():
        numero_conta = serializers.validated_data.get("numero_conta", None)
        if numero_conta:
            try:
                data_serialized = HomeReturnDataSerializer(
                    InfoAcount.objects.get(fbs_lecense__number=numero_conta),
                    many=False,
                )


                return Response(data_serialized.data, status=status.HTTP_200_OK)
            except InfoAcount.DoesNotExist:
                return Response(
                    {"error": "Conta não encontrada."}, status=status.HTTP_404_NOT_FOUND
                )
        else:
            return Response(
                {"error": "Número da conta é obrigatório."},
                status=status.HTTP_400_BAD_REQUEST,
            )




# decorators do swagger


@csrf_exempt
@api_view(["POST",])
@permission_classes([AllowAny])
def get_list_open_orders(request):
    """
       exemplo de payload:
        {
            "numero_conta": "220747665"
        }

       retorno:
       [
            {
                "fbs_lecense": 8,
                "tikect_order": "1033471413",
                "profit_order": "-7.640000000000001",
                "creater_on_bot": "2025-06-25T16:58:06Z",
                "direction": "Venda",
                "oder_result": "open",
                "order_status": "aberta",
                "currency_pair": "EURUSD",
                "modified": "2025-06-26T23:33:06.811651Z"
            },
            {
                "fbs_lecense": 8,
                "tikect_order": "1033471412",
                "profit_order": "-7.640000000000001",
                "creater_on_bot": "2025-06-25T16:58:06Z",
                "direction": "Venda",
                "oder_result": "open",
                "order_status": "aberta",
                "currency_pair": "EURUSD",
                "modified": "2025-06-26T23:33:08.297603Z"
            },
            {
                "fbs_lecense": 8,
                "tikect_order": "10334714177",
                "profit_order": "-7.640000000000001",
                "creater_on_bot": "2025-06-25T16:58:06Z",
                "direction": "Venda",
                "oder_result": "open",
                "order_status": "aberta",
                "currency_pair": "EURUSD",
                "modified": "2025-06-26T23:33:09.776213Z"
            },
            {
                "fbs_lecense": 8,
                "tikect_order": "10334714337",
                "profit_order": "-7.640000000000001",
                "creater_on_bot": "2025-06-25T16:58:06Z",
                "direction": "Venda",
                "oder_result": "open",
                "order_status": "aberta",
                "currency_pair": "EURUSD",
                "modified": "2025-06-26T23:33:12.449104Z"
            },
            {
                "fbs_lecense": 8,
                "tikect_order": "1434714337",
                "profit_order": "-7.640000000000001",
                "creater_on_bot": "2025-06-25T16:58:06Z",
                "direction": "Venda",
                "oder_result": "open",
                "order_status": "aberta",
                "currency_pair": "EURUSD",
                "modified": "2025-06-26T23:33:15.120173Z"
            }
        ]
    """
    serializers = OpenOrdersRequestSerializer(data=request.data)
    if serializers.is_valid():
        numero_conta = serializers.validated_data.get("numero_conta", None)
        if numero_conta:
            try:
                fbs_licence_id = FbsLicenses.objects.filter(number=numero_conta).first()
                get_list = Orders.objects.filter(
                    fbs_lecense=fbs_licence_id, order_status=OrderStatus.OPEN
                )
                data_serializer = OpenOrdersReturnDataSerializer(
                    get_list, many=True
                )
                if not data_serializer.data:
                    return Response(
                        {"error": "Nenhuma ordem aberta encontrada."},
                        status=status.HTTP_404_NOT_FOUND,
                    )
                return Response(data_serializer.data, status=status.HTTP_200_OK)
            except InfoAcount.DoesNotExist:
                return Response(
                    {"error": "Conta não encontrada."}, status=status.HTTP_404_NOT_FOUND
                )
        else:
            return Response(
                {"error": "Número da conta é obrigatório."},
                status=status.HTTP_400_BAD_REQUEST,
            )
    else:
        return Response(serializers.errors, status=status.HTTP_400_BAD_REQUEST)


@csrf_exempt
@api_view(["POST",])
@permission_classes([AllowAny])
def get_history_orders(request):
    """
       payload example:
        {
            "numero_conta": "220747665"
        }

       return example:
       {
            "nome_conta": "Robson Fernandes De Souza Scavazza",
            "lucro_ordens": -7.640000000000001,
            "investimento_inicial": 100.0,
            "saldo_bruto": 10081.81,
            "lisr_orders_fechadas": [
                {
                    "fbs_lecense": 8,
                    "tikect_order": "1434714337",
                    "profit_order": "-7.640000000000001",
                    "creater_on_bot": "2025-06-25T16:58:06Z",
                    "direction": "Venda",
                    "oder_result": "open",
                    "order_status": "fechada",
                    "currency_pair": "EURUSD",
                    "modified": "2025-06-26T23:55:11.835637Z"
                }
            ]
        }
    """
    serializers = HistoryOrdersRequestSerializer(data=request.data)
    if serializers.is_valid():
        numero_conta = serializers.validated_data.get("numero_conta", None)
        if numero_conta:
            try:
                fbs_licence_id = FbsLicenses.objects.filter(number=numero_conta).first()
                get_list = Orders.objects.filter(
                    fbs_lecense=fbs_licence_id, order_status=OrderStatus.CLOSED
                )
                data_serializer = HistoryOrdersReturnDataSerializer(
                    get_list, many=True
                )
                info_account = InfoAcount.objects.filter(fbs_lecense=fbs_licence_id).first()

                return_data = {
                    'nome_conta': fbs_licence_id.user.name if fbs_licence_id else None,
                    'lucro_ordens': sum(
                        float(order.profit_order) for order in get_list if order.profit_order is not None),                    'investimento_inicial': fbs_licence_id.user.initial_investment if fbs_licence_id else None,
                    'saldo_bruto': info_account.cross_balance if info_account else None,
                    'lisr_orders_fechadas': data_serializer.data
                }
                if not data_serializer.data:
                    return Response(
                        {"error": "Nenhuma ordem fechada encontrada."},
                        status=status.HTTP_404_NOT_FOUND,
                    )
                return Response(return_data, status=status.HTTP_200_OK)
            except InfoAcount.DoesNotExist:
                return Response(
                    {"error": "Conta não encontrada."}, status=status.HTTP_404_NOT_FOUND
                )
        else:
            return Response(
                {"error": "Número da conta é obrigatório."},
                status=status.HTTP_400_BAD_REQUEST,
            )
    else:
        return Response(serializers.errors, status=status.HTTP_400_BAD_REQUEST)


@csrf_exempt
@api_view(["POST",])
@permission_classes([AllowAny])
def get_profile(request):
    """
    payload example:
    {
        "numero_conta": "220747665"
    }

    return example:
    {
        "nome_conta": "Robson Fernandes De Souza Scavazza",
        "perfil_investimento": "Conservador",
        "numero_documento": "714.933.361-34"
    }
    """

    serializers = ProfileUserRequestSerializer(data=request.data)

    if serializers.is_valid():
        numero_conta = serializers.validated_data.get("numero_conta", None)
        if numero_conta:
            try:
                fbs_licence_id = FbsLicenses.objects.filter(number=numero_conta).first()
                if not fbs_licence_id:
                    return Response(
                        {"error": "Conta não encontrada."}, status=status.HTTP_404_NOT_FOUND
                    )

                data = {
                    'nome_conta': fbs_licence_id.user.name,
                    'perfil_investimento': fbs_licence_id.user.profile,
                    'numero_documento': fbs_licence_id.user.doc_number,
                }

                return Response(data, status=status.HTTP_200_OK)
            except Exception as e:
                return Response(
                    {"error": str(e)}, status=status.HTTP_500_INTERNAL_SERVER_ERROR
                )
        else:
            return Response(
                {"error": "Número da conta é obrigatório."},
                status=status.HTTP_400_BAD_REQUEST,
            )

























